﻿using System;
using System.Collections.Generic;
using System.Data;
using HMS.Class.Helper;
using HMS.Models;

namespace HMS.Class.DAL
{
    /// <summary>
    /// User DAL class.
    /// </summary>
    public class CompanyDAL : IDisposable
    {
        #region Variable Declaration

        private DBHelper databaseHelper;

        #endregion

        #region Public Methods

        /// <summary>
        /// Gets all.
        /// </summary>
        /// <param name="roleType">Type of the role.</param>
        /// <param name="searchField">The search field.</param>
        /// <param name="searchValue">The search value.</param>
        /// <param name="sortField">The sort field.</param>
        /// <param name="sortOrder">The sort order.</param>
        /// <param name="pageNo">The page no.</param>
        /// <param name="pageSize">Size of the page.</param>
        /// <returns>
        /// Returns all users.
        /// </returns>
        public List<CompanyModel> GetAll(string searchField, string searchValue, string sortField, string sortOrder, int pageNo, int pageSize)
        {
            try
            {
                this.databaseHelper = new DBHelper();

                this.databaseHelper.SetParameterToSQLCommand("@SearchField", searchField);
                this.databaseHelper.SetParameterToSQLCommand("@SearchValue", searchValue);
                this.databaseHelper.SetParameterToSQLCommand("@SortField", sortField);
                this.databaseHelper.SetParameterToSQLCommand("@SortOrder", sortOrder);
                this.databaseHelper.SetParameterToSQLCommand("@PageNo", pageNo);
                this.databaseHelper.SetParameterToSQLCommand("@PageSize", pageSize);

                IDataReader dataReader = this.databaseHelper.GetReaderByStoredProcedure("HMS_spS_Company");

                return this.GetCompanyData(dataReader);
            }
            catch
            {
                throw;
            }
            finally
            {
                this.databaseHelper.CloseConnection();
            }
        }

        /// <summary>
        /// Gets the by id.
        /// </summary>
        /// <param name="companyId">The user id.</param>
        /// <returns>Returns user by id.</returns>
        public CompanyModel GetById(long companyId)
        {
            try
            {
                this.databaseHelper = new DBHelper();

                this.databaseHelper.SetParameterToSQLCommand("@CompanyId", companyId);
                IDataReader dataReader = this.databaseHelper.GetReaderByStoredProcedure("HMS_spS_CompanyById");

                List<CompanyModel> userList = this.GetCompanyData(dataReader);

                if (userList != null && userList.Count > 0)
                {
                    return userList[0];
                }
                else
                {
                    return null;
                }
            }
            catch
            {
                throw;
            }
            finally
            {
                this.databaseHelper.CloseConnection();
            }
        }

      

        /// <summary>
        /// Saves the specified user.
        /// </summary>
        /// <param name="company">The user.</param>
        /// <returns>
        /// Returns user id if success else duplicate column name.
        /// </returns>
        public CompanyModel Save(CompanyModel company)
        {
            this.databaseHelper = new DBHelper();

            this.databaseHelper.SetParameterToSQLCommand("@CompanyName", company.CompanyName);
            this.databaseHelper.SetParameterToSQLCommand("@Address1", company.Address1);
            this.databaseHelper.SetParameterToSQLCommand("@Address2", company.Address2);
            this.databaseHelper.SetParameterToSQLCommand("@City", company.City);
            this.databaseHelper.SetParameterToSQLCommand("@State", company.State);
            this.databaseHelper.SetParameterToSQLCommand("@ZipCode", company.ZipCode);
            this.databaseHelper.SetParameterToSQLCommand("@Email", company.Email);
            this.databaseHelper.SetParameterToSQLCommand("@Phone", company.Phone);
            this.databaseHelper.SetParameterToSQLCommand("@Fax", company.Fax);
            this.databaseHelper.SetParameterToSQLCommand("@IsActive", company.IsActive);
            this.databaseHelper.SetParameterToSQLCommand("@CreatedBy", company.CreatedBy);

            IDataReader dataReader;
            CompanyModel tempUser = new CompanyModel();

            if (company.CompanyId > 0)
            {
                this.databaseHelper.SetParameterToSQLCommand("@CompanyId", company.CompanyId);
                dataReader = this.databaseHelper.GetReaderByStoredProcedure("HMS_spU_Company");
            }
            else
            {
                dataReader = this.databaseHelper.GetReaderByStoredProcedure("HMS_spI_Company");
            }

            if (dataReader != null)
            {
                using (dataReader)
                {
                    while (dataReader.Read())
                    {
                        tempUser.CompanyId = SqlHelper.GetDBLongValue(dataReader["RETURNVAL"]);
                        tempUser.DuplicateColumn = SqlHelper.GetDBStringValue(dataReader["DuplicateColumn"]);
                    }

                    return tempUser;
                }
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// Updates the multiple records.
        /// </summary>
        /// <param name="operationType">Type of the operation.</param>
        /// <param name="multiIds">The multi ids.</param>
        /// <returns>Returns 1 if success else 0.</returns>
        public int UpdateMultipleRecords(MultiOperationType operationType, string multiIds)
        {
            this.databaseHelper = new DBHelper();

            this.databaseHelper.SetParameterToSQLCommand("@MultiIds", multiIds);
            this.databaseHelper.SetParameterToSQLCommand("@OperationType", (int)operationType);

            return SqlHelper.ParseNativeInt(this.databaseHelper.GetExecuteScalarByStoredProcedure("HMS_spM_Company").ToString());
        }


        #endregion

        #region Dispose Methods

        /// <summary>
        /// Performs application-defined tasks associated with freeing, releasing, or resetting unmanaged resources.
        /// </summary>
        public void Dispose()
        {
            this.Dispose(true);
        }

        /// <summary>
        /// Releases unmanaged and - optionally - managed resources.
        /// </summary>
        /// <param name="disposing"><c>true</c> to release both managed and unmanaged resources; <c>false</c> to release only unmanaged resources.</param>
        protected virtual void Dispose(bool disposing)
        {
            if (disposing)
            {
                this.databaseHelper.Dispose();
            }
        }

        #endregion

        #region Helper Methods

        /// <summary>
        /// Gets the user data.
        /// </summary>
        /// <param name="dataReader">The data reader.</param>
        /// <returns>Returns user list.</returns>
        private List<CompanyModel> GetCompanyData(IDataReader dataReader)
        {
            if (dataReader != null)
            {
                using (dataReader)
                {
                    CompanyModel company;
                    List<CompanyModel> companyList = new List<CompanyModel>();

                    bool isDemoSite = CommonLogic.GetConfigBoolValue("IsDemoSite");

                    while (dataReader.Read())
                    {
                        company = new CompanyModel();

                        company.CompanyId = SqlHelper.GetDBLongValue(dataReader["CompanyId"]);

                        // For Demo site prevent to display default company
                        if (isDemoSite)
                        {
                            if (company.CompanyId == 1)
                                continue;
                        }

                        company.CompanyName = SqlHelper.GetDBStringValue(dataReader["CompanyName"]);
                        company.Address1 = SqlHelper.GetDBStringValue(dataReader["Address1"]);
                        company.Address2 = SqlHelper.GetDBStringValue(dataReader["Address2"]);
                        company.City = SqlHelper.GetDBStringValue(dataReader["City"]);
                        company.State = SqlHelper.GetDBStringValue(dataReader["State"]);
                        company.ZipCode = SqlHelper.GetDBStringValue(dataReader["ZipCode"]);
                        company.Phone = SqlHelper.GetDBStringValue(dataReader["Phone"]);
                        company.Fax= SqlHelper.GetDBStringValue(dataReader["Fax"]);
                        company.Email = SqlHelper.GetDBStringValue(dataReader["Email"]);
                        company.IsActive = SqlHelper.GetDBBoolValue(dataReader["IsActive"]);
                    
                        companyList.Add(company);
                    }

                    if (companyList.Count > 0)
                    {
                        dataReader.NextResult();

                        while (dataReader.Read())
                        {
                            companyList[0].TotalRecordCount = SqlHelper.GetDBIntValue(dataReader["TotalRecordCount"]);
                        }
                    }

                    return companyList;
                }
            }
            else
            {
                return null;
            }
        }

        #endregion
    }
}